<?php

namespace Noptin\WooCommerce\Triggers;

/**
 * Subscription before end trigger.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Fires before a subscription ends.
 *
 * @since 1.0.0
 */
class Subscription_Before_End extends Subscription_Before_Renewal {

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'woocommerce_subscription_before_end';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'WooCommerce Subscription > Before End', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'X days before a subscription ends', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_rule_description( $rule ) {
		$settings = $rule->trigger_settings;

		/* translators: %s Renewal days */
		return $this->get_days( $settings, __( 'Days before end: %s', 'noptin-woocommerce' ) );
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array_merge(
			array(
				'days' => array(
					'el'          => 'input',
					'label'       => __( 'Days before end', 'noptin-woocommerce' ),
					'description' => __( 'Enter a comma-separated list of days before end for which this trigger will fire.', 'noptin-woocommerce' ),
					'placeholder' => '1, 3, 7',
				),
			),
			parent::get_settings()
		);

	}

	/**
	 * Fetch subscriptions for the given number of days.
	 *
	 * @param int $days The days before expiry.
	 * @return int[] $subscription_ids The subscription ids.
	 */
	public function get_subscriptions_for_days( $days ) {

		return get_posts(
			array(
				'post_type'      => 'shop_subscription',
				'post_status'    => array( 'active', 'pending-cancel' ),
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => array(
					array(
						'key'     => wcs_get_date_meta_key( 'end' ),
						'value'   => gmdate( 'Y-m-d 00:00:00', strtotime( "+{$days} days" ) ),
						'compare' => '>=',
						'type'    => 'DATETIME',
					),
					array(
						'key'     => wcs_get_date_meta_key( 'end' ),
						'value'   => gmdate( 'Y-m-d 23:59:59', strtotime( "+{$days} days" ) ),
						'compare' => '<=',
						'type'    => 'DATETIME',
					),
				),
			)
		);
	}

}
